; Source code for the game "1K-MINI-BDASH"
; written 2008/05/25 by ssdsa (Simon Stelling, S.E.S./Genesis*Project)
; for the "1024 Bytes Game Compo (2nd Edition)" of www.forum64.de
;
; Assembler: "C64ASM.py" (version "V1.0.003 2008-05-23")
; Target: Commodore C64 (PAL).
;
; --
;
; This game called "1K-MINI-BDASH" is a tiny Boulder Dash clone
; that fits within 1024 bytes. Unbelievable!
; The game features 4 different caves (A, B, C, D)
; with 256 different levels each - so you get 1024 screens
; of game play.
;
; Written by ssdsa (Simon Stelling, S.E.S./Genesis*Project) 2008/05/25
; for the "1024 Bytes Game Compo (2nd Edition)" of www.forum64.de
;
; If you have any questions or problems
; concerning this game, feel free to
; contact me: simon.stelling@gmx.de
;
; For compo details see this forum thread:
; http://www.forum64.de/wbb3/index.php?page=Thread&threadID=22110
;
; Compo rules:
; Game has to run on a plain C64 with joystick, keyboard and tv.
;  (It is OK if the game runs on PAL video mode only.)
; Game file on disk maximum size is 1026 bytes
;  (including the 2 bytes for the loading adress).
; Game start by LOAD"GAME",8,1 followed by RUN.
;
;
; Game Instructions:
;
; You are Rockford. You dig through dirt
; caves collecting diamonds. Watch out for
; falling boulders and deadly fireflies.
; You need to collect 10 diamonds, then
; the cave exit opens. Reach the cave exit
; to win the cave and proceed to the next
; cave. There is no time limit.
; If you get stuck in a cave, press
; run/stop to restart the cave.
;
; Each time you finish a cave, you get an
; extra life. The maximum number of lifes
; you can get is 10. If you have got 10
; lifes and win another cave, you don't
; get another extra live, but you get a
; bright white 10th Rockford life
; indicator as a special reward.
;
; Move Rockford with Joystick in port 2.
;
; You can try to push boulders to the left
; or to the right, but they are heavy and
; there has to be free space behind them.
;
; When you press fire while moving,
; Rockford acts as if he moves one step
; (digs through dirt, collects diamond,
; tries to push a boulder) but without
; actually moving. This can be handy when
; things get tricky.
;
; The game is over when you lost your last
; life. Then, press run/stop or fire on
; joystick to restart the whole game,
; starting in cave A on the first level
; again.
;
; Every 4th cave, the game level increases
; by 1, giving you 256 different levels of
; all 4 caves, so there are 1024 different
; screens of game play for you!
;
; As a special feature, in the first level
; the caves A, B and C are 99% similar to
; the original Boulder Dash 1 caves.
; Cave D is unique to this game, though.
;
; Have fun!
;
;
; ========================================================================

PC $0002
zp_clear::

zp_cave_draw:           .wo 0   ; low-byte is always kept zero
zp_cave_scan:           .wo 0
zp_screen_scan:         .wo 0
zp_colram_scan:         .wo 0
zp_rand_seed_A:         .by 0
zp_rand_seed_B:         .by 0
zp_tmp_a:               .by 0   ; used for several temp values
zp_tmp_b:               .by 0   ; used for temp value for predictable random numbers
zp_line_step:           .by 0
zp_rockford_seen_alive: .by 0
zp_inbox_delay:         .by 0
zp_diamonds_needed:     .by 0   ; = 0 --> outbox is open
zp_flash_border:        .by 0
zp_current_cave:        .by 0
zp_current_level:       .by 0

zp_dummy_1:             .by 0
zp_dummy_2:             .by 0
zp_dummy_3:             .by 0
zp_dummy_4:             .by 0

decode_cave_rand_obj:   .by $01 ; object $01 = dirt
                        .by 0,0,0
decode_cave_rand_threshold:     ; first byte of  decode_cave_rand_threshold  is never read
                        .by 0   ; so this is the last byte of  decode_cave_rand_obj
                        .by 0,0,0,0

cave_colors:      .by 0     ; color index 0: ($D021) black background (fixed)
                  .by 8     ; color index 1: ($D022) titan wall color
                  .by 9     ; color index 2: ($D023) dark dirt color
                  .by 7     ; color index 3: ($D024) diamond body color
zp_copy_to::
                  .by 1     ; color index 4: diamond flash color / firefly white color (fixed)
                  .by 15    ; color index 5: rockford color / wall color

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

DEFINE NUMBER_OF_CAVES 4

cave_2_sp:              .by SP_FOR_CAVE_A
                        .by SP_FOR_CAVE_D
                        .by SP_FOR_CAVE_C
                        .by SP_FOR_CAVE_B

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

zp_lifes_left:          .by INITIAL_LIFES

DEFINE INITIAL_LIFES  3
DEFINE MAX_LIFES     10

; - - - - - - - - - - - - - - - - -

; Explosions either occur at the current object's position, or below the current object.
; Positions to explode are then: 0, 1, 2,  40, 41, 42,  80, 81, 82
;                         resp.:           40, 41, 42,  80, 81, 82, 120, 121, 122
explosion_positions:  ; table of 12 bytes
; first 3 bytes used only when object explodes below
; last 3 bytes used only when object itself explodes
                        .by 120,121,122
                        .by  80, 82, 41
joy_2_dir:  .by  42, 40, 81, 1
                        .by   0,  2

zp_firefly_dir1:        .by 42
zp_firefly_dir2:        .by 1, 40, 81, 42

zp_tab_39_43:           .by 39, 43

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

charset_src::

    .by $EB,$BE,$7F,$ED,$BB,$F7,$DD,$F7  ; char $00: dirt (and also explosion3)

    .by $3E,$3E,$1C,$7F,$5D,$3C,$36,$77  ; char $01: rockford

;   .by $77,$00,$DD,$00,$77,$00,$DD,$00  ; char $02: wall
    .by %.XXX.XXX
    .by %........
    .by %XX.XXX.X
    .by %........
    .by %.XXX.XXX
    .by %........
    .by %XX.XXX.X
    .by %........

    .by $FF,$E7,$C3,$81,$00,$81,$C3,$E7  ; char $03: diamond

charset_firefly:
;   .by $00,$00,$3C,$3C,$3C,$3C,$00,$00  ; char $04: firefly
    .by %00000000
    .by %00000000
    .by %00111100
    .by %00111100
    .by %00111100
    .by %00111100
    .by %00000000
    .by %00000000

    .by $00,$44,$00,$00,$00,$44,$00,$00  ; char $05: titan (also inbox and outbox)

    .by $3C,$7E,$FF,$DF,$BF,$D5,$6A,$3C  ; char $06: boulder

    .by $00,$62,$16,$C5,$5A,$63,$94,$00  ; char $07: explosion2

DEFINE charset_char_offset  (charset_src/8)

ASSERT (charset_src&7) == 0
ASSERT charset_char_offset = 8

;   Charset codes:
;
;       Name:       char:       $D800  $D021/2/3
;       Space       (any)        0(!)   D021 = 0
;   0   Dirt        dirt         9      D022 = 8
;    0  Explosion3  dirt         *      D021 = 0
;   1   Rockford    rockford    15      D021 = 0
;   2   Wall        wall        15      D021 = 0
;   3   Diamond     diamond      0(!)   D024 = 1/7
;   4   Firefly     firefly*     8      D023 = 9/1  *: Several anim steps
;   5   Titan       titan        1      D022 = 8
;    5  Fade titan wall         15      D022 = 8
;    5  Inbox       titan*       0      D022 = 8    *: Open inbox alternating titan/space
;    5  Explosion1  explosion1   *      D021 = 0    *: Explode to space: 1,1,7,10,2; Rockford birth: 15, 15, 15
;   6   Boulder     boulder      8      D021 = 0
;   7   Explosion2  explosion2   *      D021 = 0

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

; For every kind of object, we have to store three different pieces of information:
;  - charset code  (0 .. 7)
;  - background color mode  ($00, $40, $80 or $C0)
;  - color index for fetching color value for $D800  (0 .. 5)
; To save storage space, we store all these three pieces in a single byte,
; like this:
; Bits 7 and 6:  background color mode
; Bits 5, 4 and 3:  color index,  shifted by 3 bits.
; Bits 2, 1 and 0:  charset code.

object_2_char:
    .by $00+(0<<3) ; object $00: space (any value from $00 to $3F is OK since $D800=0)
    .by $40+(2<<3) ; object $01: dirt
    .by $45+(4<<3) ; object $02: titan wall
    .by $00+(0<<3) ; object $03: (unused)
    .by $45+(5<<3) ; object $04: fade in/out titan wall
    .by $05+(2<<3) ; object $05: explode to space 4  (-->space)
    .by $07+(1<<3) ; object $06: explode to space 3  (-1)
    .by $00+(3<<3) ; object $07: explode to space 2  (-1)
    .by $07+(4<<3) ; object $08: explode to space 1  (-1)
    .by $05+(4<<3) ; object $09: explode to space 0  (-1)
    .by $01+(5<<3) ; object $0A: Rockford
    .by $01+(5<<3) ; object $0B: Rockford stf        (-1)
    .by $00+(4<<3) ; object $0C: Rockford birth 3    (-1)
    .by $07+(5<<3) ; object $0D: Rockford birth 2    (-1)
    .by $05+(5<<3) ; object $0E: Rockford birth 1    (-1)
    .by $02+(5<<3) ; object $0F: wall
    .by $C3+(0<<3) ; object $10: Diamond
    .by $C3+(0<<3) ; object $11: Diamond stf (-1)
    .by $06+(1<<3) ; object $12: Boulder
    .by $06+(1<<3) ; object $13: Boulder stf (-1)
    .by $C3+(0<<3) ; object $14: Diamond falling
    .by $C3+(0<<3) ; object $15: Diamond falling stf (-1)
    .by $06+(1<<3) ; object $16: Boulder falling
    .by $06+(1<<3) ; object $17: Boulder falling stf (-1)
    .by $84+(1<<3) ; object $18: Firefly0
    .by $45+(4<<3) ; object $19: inbox        (blink; -->Rockford birth 1)
    .by $84+(1<<3) ; object $1A: Firefly1
    .by $45+(4<<3) ; object $1B: outbox       (blink when opened)
    .by $84+(1<<3) ; object $1C: Firefly2
    .by $84+(1<<3) ; object $1D: Firefly2 stf  (-1)
    .by $84+(1<<3) ; object $1E: Firefly3
    .by $84+(1<<3) ; object $1F: Firefly3 stf  (-1)

     ; charset:
     ; char $00: dirt (and also explosion3)
     ; char $01: rockford
     ; char $02: wall
     ; char $03: diamond
     ; char $04: firefly
     ; char $05: titan (and also inbox and outbox and explosion1)
     ; char $06: boulder
     ; char $07: explosion2

     ; colors:
     ;  0    ; color index 0: ($D021) black background (fixed)
     ;  8    ; color index 1: ($D022) titan wall color
     ;  9    ; color index 2: ($D023) dark dirt color
     ;  7    ; color index 3: ($D024) diamond body color
     ;  1    ; color index 4: diamond flash color / firefly white color (fixed)
     ; 15    ; color index 5: rockford color / wall color (fixed)

; - - - - - - - - - - - - - - - - -

; For every kind of object, we have to store the adress of the handler routine
; that handles this kind of object.
; Since all handler routines are located between $0800 and $08FF,
; the high byte of all handler adresses is #$08, so we just have to
; store the different low bytes.

object_2_handler:
  .by <(handle_nop)             ; object $00: space
  .by <(handle_nop)             ; object $01: dirt
  .by <(handle_nop)             ; object $02: titan wall
  .by <(handle_nop)             ; object $03: (unused)
  .by <(handle_nop)             ; object $04: fade in/out titan wall
  .by <(handle_to_space)        ; object $05: explode to space 4  (-->space)
  .by <(handle_minus_1)         ; object $06: explode to space 3  (-1)
  .by <(handle_minus_1)         ; object $07: explode to space 2  (-1)
  .by <(handle_minus_1)         ; object $08: explode to space 1  (-1)
  .by <(handle_minus_1)         ; object $09: explode to space 0  (-1)
  .by <(handle_rockford)        ; object $0A: Rockford
  .by <(handle_minus_1)         ; object $0B: Rockford stf        (-1)
  .by <(handle_minus_1)         ; object $0C: Rockford birth 3    (-1)
  .by <(handle_minus_1)         ; object $0D: Rockford birth 2    (-1)
  .by <(handle_minus_1)         ; object $0E: Rockford birth 1    (-1)
  .by <(handle_nop)             ; object $0F: wall
  .by <(handle_boulder)         ; object $10: Diamond
  .by <(handle_minus_1)         ; object $11: Diamond stf (-1)
  .by <(handle_boulder)         ; object $12: Boulder
  .by <(handle_minus_1)         ; object $13: Boulder stf (-1)
  .by <(handle_boulder_falling) ; object $14: Diamond falling
  .by <(handle_minus_1)         ; object $15: Diamond falling stf (-1)
  .by <(handle_boulder_falling) ; object $16: Boulder falling
  .by <(handle_minus_1)         ; object $17: Boulder falling stf (-1)
  .by <(handle_firefly)         ; object $18: Firefly0
  .by <(handle_inbox)           ; object $19: inbox        (blink; -->Rockford birth 1)
  .by <(handle_firefly)         ; object $1A: Firefly1
  .by <(handle_outbox)          ; object $1B: outbox       (blink when opened)
  .by <(handle_firefly)         ; object $1C: Firefly2
  .by <(handle_minus_1)         ; object $1D: Firefly2 stf  (-1)
  .by <(handle_firefly)         ; object $1E: Firefly3
  .by <(handle_minus_1)         ; object $1F: Firefly3 stf  (-1)

; - - - - - - - - - - - - - - - - -

zp_copy_end:

zp_end:

ASSERT zp_end < $0100

DEFINE zp_clear_length (zp_copy_to-zp_clear)
ASSERT zp_clear_length <= INIT_MAX_COPY

DEFINE zp_copy_length (zp_copy_end-zp_copy_to)
ASSERT zp_copy_length <= INIT_MAX_COPY

ASSERT zp_copy_to+INIT_MAX_COPY < $0100

; ========================================================================
; ========================================================================
; ========================================================================

DEFINE file_start $0801

PC file_start-2
 .wo file_start  ; generate low- and high-byte of start address as first 2 bytes of file

; BASIC start line:   19519 SYS2438
 .wo $080b
 .wo 19519
 .by $9e        ; "SYS"
 .by $30+GAME_START_DIGIT_1
 .by $30+GAME_START_DIGIT_2
 .by $30+GAME_START_DIGIT_3
 .by $30+GAME_START_DIGIT_4
 .by 0,0,0

; ========================================================================
; ========================================================================
; ========================================================================

first_handle::
handle_boulder::        ldy #81
                        lda (zp_cave_scan),Y
                        beq _start_falling_down

                        ; is the object below the current object round?
                        ; object is round means: object is one of these: ($0E), $0F, $10, $11, $12, ($13)
                        ; We don't know if C=0 or C=1 here
                        ; and we don't want to use a  SEC  in order to save one byte.
                        sbc #$0E  ; ($0E), $0F, $10, $11, $12, ($13) --> ($00), $01, $02, $03, $04, ($05)
                                  ;                              or  --> ($FF), $00, $01, $02, $03, ($04)
                        cmp #$05  ; is $00 .. $04?
                        bcs handle_rts1 ; is not round

                        ; can roll left?
                        dey
                        lda (zp_cave_scan),Y
                        ldy #40
                        ora (zp_cave_scan),Y
                        beq _roll_left

                        ; can roll right?
                        ldy #82
                        lda (zp_cave_scan),Y
                        ldy #42
                        ora (zp_cave_scan),Y
                        bne handle_rts1 ; cannot roll left nor right
_roll_left:
_roll_right:
_start_falling_down:
_keep_falling_down:
                        txa
                        ora #4     ; $10 to $14, resp. $12 to $16
                        jsr set_object_when_stf_plus_1
handle_to_space:
set_space_at_41:
                        lda #0     ; $00: space
                        bpl set_object_at_41   ; branches always
 ; object $00: space
 ; object $0F: wall
 ; object $10: Diamond
 ; object $12: Boulder
 ; object $14: Diamond falling
 ; object $16: Boulder falling

; ========================================================================

handle_boulder_falling:
                        ldy #81
                        lda (zp_cave_scan),Y
                        beq _keep_falling_down

; We don't roll falling boulders in order to save bytes.
; When they fall and hit something and could roll, they simply stop falling,
; and then roll. That's not exactly like the original BD1, but it's very close.

                        ; Check if falling boulder hits a Firefly or Rockford:

                        ; all objects that have bit 0 set --> don't crush
                        lsr
                        bcs _stop_falling

                        ; Rockford (object $0A) --> crush
                        cmp #$0A/2
                        beq _crush

                        ; Any object >= $18 is a Firefly --> crush
                        cmp #$18/2
                        bcs _crush

_stop_falling:
                        txa
                        and #$12   ; object $14 to object $10, resp. $16 to $12
                        bpl set_object_at_41  ; branches always

_crush:                 ; Explode 3x3 area
                        lda #explosion_positions
explode_at:
                        sta _read_from+1
                        lda #8
                        sta zp_tmp_a
_crush_loop:                ldx zp_tmp_a
_read_from:                 ldy explosion_positions,X

                            ; Check if there is titan wall:
                            lda (zp_cave_scan),Y
                            cmp #$02   ; object $02: titan wall
                            beq _dont_explode_titan

                            lda #$08   ; object $08: explode to space 1
                            jsr set_object_when_stf_plus_1
_dont_explode_titan:

                            dec zp_tmp_a
                            bpl _crush_loop

handle_rts1:            rts

; ========================================================================

handle_firefly::
                        txa

      ;;;               lsr
      ;;;               and #3
                      ; ALR #6 ; illegal opcode, just like AND #6 followed by LSR
                        .by $4B, 6

                        sta zp_tmp_a

                        ldx #3
                        lda #$0A    ; object $0A: Rockford
_search_for_rockford:   ldy joy_2_dir,X
                        cmp (zp_cave_scan),Y
                        beq explode
                        dex
                        bpl _search_for_rockford

                        ldx zp_tmp_a
                        ldy zp_firefly_dir1,X
                        lda (zp_cave_scan),Y
                        bne _try_move
                        dex

_move:                  txa
                        and #3
                        asl
                        ora #$18   ; object $18: Firefly0;  $1A: Firefly1  ; $1C: Firefly2  ; $1E: Firefly3
                        pha
                        tya
                        pha
                        jsr set_space_at_41
                        pla
                        tay
                        pla
                        bne set_object_when_stf_plus_1 ; branches always

_try_move:              ldy zp_firefly_dir2,X
                        lda (zp_cave_scan),Y
                        beq _move

                        inx
                        ldy #41
                        bpl _move  ; branches always

; ========================================================================

explode:                lda #explosion_positions+3
                        bne explode_at    ; branches always

; ========================================================================

set_object_when_stf_plus_1:
                        cpy #42
                        adc #0  ; plus 1 when Y >= 42 since C=1 then; otherwise plus 0
                        bpl set_object ; branches always

; ========================================================================

handle_minus_1:         dex
                        txa

; ------------------------------------------------------------------------

set_object_at_41:       ldy #41

; ------------------------------------------------------------------------

set_object:             ; store object in cave:
                        sta (zp_cave_scan),Y
                        tax

                        ; update low byte of screen ptr and colram ptr from cave ptr:
                        lda zp_cave_scan
                        sta zp_screen_scan
                        sta zp_colram_scan

                        ; update high byte of screen ptr and colram ptr from cave ptr:
                        lda zp_cave_scan+1
                        eor #(>(cave_buffer))^(>(screen))
                        sta zp_screen_scan+1
                        eor #(>(colram))^(>(screen))
                        sta zp_colram_scan+1

                        ; store char:
                        lda object_2_char,X
                        and #$C7
                        ora #charset_char_offset
                        sta (zp_screen_scan),Y

                        ; store color:
                        lda object_2_char,X
        ;;;             and #$38
        ;;;             lsr
                      ; ALR #$38    ; illegal opcode, just like AND #$38 followed by LSR
                        .by $4B, $38

                        lsr
                        lsr
                        tax
                        lda cave_colors,X
                        sta (zp_colram_scan),Y

handle_rts:             rts

; ========================================================================

handle_inbox::          lda #$0E    ; object $0E: Rockford birth 1
                        dec zp_inbox_delay
                        beq set_object

; ------------------------------------------------------------------------

blink_object_in_X::     lda object_2_char,X
     ; in order to blink, we have to alternate these charset codes:

     ; char $45: titan
DEFINE blink_value_titan   $45+(4<<3)

     ; any value from $00 to $3F: space
DEFINE blink_value_space   $00+(0<<3)

                        eor #blink_value_titan^blink_value_space
                        sta object_2_char,X

                        txa
bpl_to_set_object:      bpl set_object     ; branches always

; ========================================================================

handle_outbox::
; When zp_diamonds_needed = 0, then blink outbox:
                        lda zp_diamonds_needed
                        beq blink_object_in_X

; Otherwise the outbox is still closed.
; Check if the outbox graphics is really closed:
                        lda object_2_char,X
ASSERT blink_value_space = 0
    ;;;                    eor #blink_value_space
                        beq blink_object_in_X    ; blink once to close outbox graphics

                        rts

; ========================================================================

handle_nop::
handle_diamond_anim::   bne animate_diamond_color  ; branches always

; ========================================================================

ASSERT >(first_handle)  =  >(last_handle)
last_handle::
handle_rockford::
                        ; check for run/stop key to restart cave:
                        lda $DC01
                        bpl explode

    ; here Y=41
                        sty zp_rockford_seen_alive   ; Update counter "Rockford seen alive"

                        lda $DC00
; Bit 0..3: Direction up, down, left, right
; Bit 4: Fire

                        ; Y=41 here
                        ldx #3
_joy_loop:              lsr
                        bcs _skip_this_dir
                        ldy joy_2_dir,X
_skip_this_dir:         dex
                        bpl _joy_loop

DEFINE _rockford_stays   handle_rts

                        cpy #41 ; joystick centered?
                        beq _rockford_stays

                        lda (zp_cave_scan),Y
 ; object $00: space
 ; object $01: dirt
 ; object $10: Diamond
 ; object $12: Boulder
 ; object $1B: outbox
                        ; is space or dirt? --> Rockford can move
                        cmp #2
                        bcc _rockford_moves

                        ; diamond? --> Rockford collects diamond, and can move
                        cmp #$10  ; object $10: Diamond
                        beq _collect_diamond

                        ; outbox --> cave won if open
                        cmp #$1B  ; object $1B: outbox
                        beq _check_outbox

                        cmp #$12  ; object $12: Boulder
                        bne _rockford_stays
           ; boulder (and dir = left or right) --> try to push, can move if push successful
                        ; check direction:
                        tya
                        ; now A=40 when left, A=42 when right,  A=1 when up, A=81 when down.
                        lsr       ; up or down --> C=1;  left or right --> C=0
                        bcs _rockford_stays
           ; OK, dir = left or right
                        sty zp_tmp_a
                        ; now A=20 when left, A=21 when right
                        ; we want to get Y=39 when left, Y=43 when right.
                        ; we simply use a lookup-table:
                        tax
                        ldy zp_tab_39_43-20,X
                        lda (zp_cave_scan),Y
                        bne handle_rts2     ; Rockford cannot push blocked boulder. Rockford cannot move.

                        ; Check for random number because we want pushing to succeed in (approx.) 1 out of 4 tries.
                        ; How do we do this using only an absolute minimum of bytes?
                        ; We have Y=39 or Y=43. In PAL systems, $D012 runs from 0 to 311
                        ; but since $D012 contains bits 0..7 only, $D012 actually runs from 0 to 255
                        ; and then from 0 to 55. When we compare Y to $D012, we get a
                        ; (39+39) : 312  resp.  (43+43) : 312 chance,
                        ; that  0.25     resp.  0.27.
                        ; This is really close to a 1 : 4 chance.
                        cpy $D012
                        bcc handle_rts2 ; Bad luck at pushing. Rockford cannot move.

                        ; store boulder at new boulder position:
                        lda #$12    ; object $12: Boulder
                        jsr set_object_when_stf_plus_1
                        ldy zp_tmp_a
                        bne _rockford_moves    ; branches always

_collect_diamond:       ; Increase score. Open outbox when needed number of diamonds reached.
                        ldx zp_diamonds_needed
                        beq _rockford_moves   ; already enough diamonds; ignore more diamonds

                        lda #$C3+charset_char_offset    ; char $C3: diamond
                        sta screen_diamonds-1,X

                        dec zp_diamonds_needed
                        bne _not_enough_yet
; Initiate a single short white flash in the border:
                        lda #$07
                        sta zp_flash_border
_not_enough_yet:

_rockford_moves:        ; Check joy fire:
                        lda $DC00
                        and #$10  ; joy fire pressed?

                        ; If joy fire is pressed:
                        ;  We have to store space at destination and don't move Rockford.
                        ;  A=0 here, that's handy because  object $00: space
                        beq bpl_to_set_object

; move Rockford (fire not pressed):
                        lda #$0A   ; object $0A: Rockford
                        jsr set_object_when_stf_plus_1
                        jmp set_space_at_41

; ========================================================================

animate_diamond_color:  jsr _twice

_twice:                 lda $D024
                        eor cave_colors+3   ; color index 3: diamond body color
                        eor cave_colors+4   ; color index 4: diamond flash color
                        sta $D024
handle_rts2:            rts

; ========================================================================

; and here is the rest of handle_rockford:

_check_outbox:
                        ; Check if outbox is open:
                        lda zp_diamonds_needed
                        bne handle_rts2  ; Outbox is still closed, so Rockford cannot move.

; outbox reached, cave won:
cave_won_A0:
                        ; here A = 0
                        sta zp_cave_scan ; init zp_cave_scan with #0 for next fade_in

                        ; continue with next cave:
                        dec zp_current_cave
                        bpl _skip_wrap

                        ldy #NUMBER_OF_CAVES-1
                        sty zp_current_cave

                        ; Increase level:
                        inc zp_current_level

_skip_wrap:

; The player gets an extra life after every completed cave.

                        ; Extra life (up to 10 lifes)
                        ldx zp_lifes_left
                        cpx #MAX_LIFES
                        ; here still A = 0
                        beq _no_more_new_lifes

                        inc zp_lifes_left
                        lda #$0F^$01 ; color new life Rockford in status bar light grey ($0F)
                        inx

_no_more_new_lifes:     eor #$01     ; color current life Rockford in status bar white ($01)
                        sta colram-1,X

; If we want to make the game harder and award an extra life only when
; the level increases, we could define the label _skip_wrap at the
; alternative position indicated below:
;;;_skip_wrap:

                        ; Remove return address from stack since we don't want to trash our cave data
                        ; which is stored on the stack
                        pla
                        pla

                        ; continue with next cave
                        bne start_cave  ; branches always

; ========================================================================
; ========================================================================
; ========================================================================

game_start::

DEFINE GAME_START_DIGIT_1   (game_start/1000)%10
DEFINE GAME_START_DIGIT_2   (game_start/ 100)%10
DEFINE GAME_START_DIGIT_3   (game_start/  10)%10
DEFINE GAME_START_DIGIT_4   (game_start/   1)%10

                        sei

DEFINE INIT_MAX_COPY $D7

                        ldx #INIT_MAX_COPY

; Init $D011 with #$57:
ASSERT (INIT_MAX_COPY&$7F) = $57
                        stx $D011

        ; X = #$D7 here
        ; set SP to #$D6:
                        txs
                        pha

_init_zp_data:
      ; draw 10 Rockfords, and set text color of top two screen lines to black:
                        lda #$01+charset_char_offset    ; char $01: rockford
                        sta screen-1,X
                        lda #0
                        sta colram-1,X

      ; set text color to light grey (color 15) for first 3 Rockfords:
                        cpx #INITIAL_LIFES+1
                        bcs _skip_Rockford
                        dec colram-1,X    ; $F0 -> $EF, so color 15.
_skip_Rockford:
      ; clear zero page:
                        sta zp_clear-1,X

      ; copy initial data to zero page:
                        lda zp_data-1,X
                        sta zp_copy_to-1,X

      ; copy initial data to stack:
                        lda cave_data_for_stack_src-1,X
                        pha

                        dex
                        bne _init_zp_data
      ; now SP = #$FF.

; We know that A = #value_D018/2 here, so we can save one byte:
                        asl
                ;;;     lda #value_D018
                        sta $D018

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

DEFINE screen           $3C00
DEFINE colram           $D800
DEFINE cave_buffer      $7C00
DEFINE charset          $0000

DEFINE value_D018       ((screen/1024)<<4)+(charset/1024)

DEFINE cave_scan_start_adr (((cave_buffer+40+40+40)-41)-39)
DEFINE adr_titan_border_top_left (cave_buffer+40+40)
DEFINE adr_titan_border_bottom_left (cave_buffer+(23*40))

DEFINE INITIAL_DIAMONDS_NEEDED 10
DEFINE screen_diamonds  (screen+40)-INITIAL_DIAMONDS_NEEDED

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

start_cave::
                        lda #$A9    ; #$A9 == fade out to titan wall
                        jsr fade_in

; set up current cave (including colors), and
; draw current cave (random objects and fixed objects):
                        ldy zp_current_cave
                        ldx cave_2_sp,Y
                        jsr setup_cave

; init number of diamonds needed,
; and draw diamonds in dark dirt color in status bar
                        ldx #INITIAL_DIAMONDS_NEEDED
                        stx zp_diamonds_needed
                        dex
                        lda #$43+charset_char_offset    ; char $43: charset for diamond, but in titan color
_draw_dark:             sta screen_diamonds,X
                        dex
                        bpl _draw_dark
                        ; now X = #$FF

; draw titan border:
   ;;                   ldx #$FF  ; X = #$FF already, see above
                        jsr draw_objects

                        lda #$B1   ; #$B1 == fade in current cave
                        jsr fade_in

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

in_game_loop::

; Is key "2" pressed?
_modify:                lda #$FF     ; modify this into  lda #8  to enable cheat mode. Then press key "2" to immediately win cave.
                        and $DC01
                        bne _no_cheat_skip
; Cheat mode: Immediately win current cave.
                        ; A = 0 here
                        jsr cave_won_A0  ; never returns
_no_cheat_skip:

DEFINE POKE_8_INTO_HERE_FOR_CHEAT _modify+1


; init cave scan:
                        lda #>(cave_scan_start_adr)
                        sta zp_cave_scan+1
                        lda #<(cave_scan_start_adr)
                        sta zp_cave_scan

ASSERT ((<(cave_scan_start_adr))&$07) = 0
; flash border when outbox opens:
                        lsr zp_flash_border
                        rol
                        sta $D020

; animate Firefly charset and color:
                        ldx #7
_anim_firefly_charset:  lda charset_firefly,X
                        eor #$FF
                        sta charset_firefly,X
                        dex
                        bpl _anim_firefly_charset

                        asl
                        bcc _anim_firefly_done

                        lda $D023
                        eor cave_colors+2   ; color index 2: dark dirt color
                        eor cave_colors+3   ; color index 3: diamond body color
                        sta $D023   ; $D023: alternating Firefly color
_anim_firefly_done:

; start cave scan:
cave_scan_loop::
                        ldy #41

      ;;;               lda (zp_cave_scan),Y
      ;;;               tax
                      ; LAX (zp_cave_scan),Y  ; illegal opcode, works like LDA (..),Y followed by TAX
                        .by $B3, zp_cave_scan

                        lda object_2_handler,X
                        sta _jsr+1
_jsr:                   jsr handle_boulder

                        inc zp_cave_scan
                        bne cave_scan_loop

ASSERT (cave_buffer+$0400) = $8000
                        inc zp_cave_scan+1      ; End when $8000 reached
                        bpl cave_scan_loop
; end of cave scan.

                        ; Check if Rockford is dead and already lost his current life:
                        lda zp_rockford_seen_alive
                        beq _rockford_life_already_lost

                        ; Check if Rockford was seen alive recently:
                        dec zp_rockford_seen_alive
                        bne in_game_loop
; cave lost:
                        ldx zp_lifes_left
                        lda #$0B         ; set dark grey ($0B) color for dead Rockford in status bar
                        sta colram-1,X

                        dec zp_lifes_left
                        ; if there is at least one life left, restart current cave:
                        bne start_cave

_rockford_life_already_lost:
; game lost:

; The player has to press run/stop or joy fire
;  to restart the whole game after he lost his last life.
; As long as the player doesn't press run/stop or joy fire, we continue the game loop:
                        lda $DC00
                        and #$10  ; joy fire pressed?
                        beq _restart_game

                        lda $DC01 ; run/stop pressed?
                        bmi in_game_loop

_restart_game:          jmp game_start   ; restart whole game

; ========================================================================

; Input: X = initial value for SP to access begin of cave data on stack
setup_cave::            txs

; fetch and set initial random seed:
                        pla
                        eor zp_current_level
                        sta zp_rand_seed_A

; fetch and set initial thresholds, random objects and cave colors:
                        ldx #12
_set_threshold:         pla
                        sta decode_cave_rand_obj,X
     ;;                 cpx #9
     ;;                 bcc _skip_vic
                        ; store colors in $D021 .. $D024
                        sta $D021-9,X  ; this also overwrites $D019 .. $D020, but we don't care
;;_skip_vic:

; init  decode_cave_rand_obj  with  $01: dirt
                        stx decode_cave_rand_obj

                        dex
                        bne _set_threshold

      ; set $D020 to black:
                        stx $D020

; clear hi-byte of random seed:
                        stx zp_rand_seed_B

; generate random cave objects:
ASSERT <(cave_buffer) = 0
                        lda #>(cave_buffer)
                        sta zp_cave_draw+1
                        ldy #3*40
_loop:
                        ; Get predictable random number:
                        lda zp_rand_seed_B

         ;;;            and #1
         ;;;            lsr
                      ; ALR #1 ; illegal opcode, just like AND #1 followed by LSR
                        .by $4B, 1

                        ror
                        tax
                        lda zp_rand_seed_A
                        lsr
                        sta zp_tmp_b
                        lda zp_rand_seed_A

         ;;;            and #1
         ;;;            lsr
                      ; ALR #1 ; illegal opcode, just like AND #1 followed by LSR
                        .by $4B, 1

                        ror
                        ; here C = 0
                        adc zp_rand_seed_A
                        adc #$13
                        sta zp_rand_seed_A
                        txa
                        adc zp_rand_seed_B
                        adc zp_tmp_b
                        sta zp_rand_seed_B
                        ; Now A is random.
                        ldx #4
_loop_search:               cmp decode_cave_rand_threshold,X
                            bcc _index_found
                            dex
                            bne _loop_search
_index_found:           lda decode_cave_rand_obj,X
                        sta (zp_cave_draw),Y

   ; Since fade out, decode cave and fade in
   ; is so fast, we have to slow things down a little bit
   ; so the player has time to prepare for the next level:
_slow_wait:             inx
                        bne _slow_wait

                        iny
                        bne _loop

ASSERT (cave_buffer+$0400) = $8000
                        inc zp_cave_draw+1  ; End when $8000 reached
                        bpl _loop

                        tsx ; fall through to next routine to draw cave specific objects

; ------------------------------------------------------------------------

; Input: X = begin of draw commands on stack
;  Each command is 3 bytes:
;   b1  =  length (bits 0..5) and direction (bit 7: 0=right; 1=down)
;   b2  =  adr (lo) of start pos
;   b3  =  bits 0..1: adr (hi) of start pos
;          bits 2..6: object to draw (2x LSR needed)
;          bit 7: if set, quit drawing after this command

draw_objects::          txs
_continue:
                        ldy #1
                        pla
                        bpl _right
; In order to save 2 bytes, we draw all lines that have direction=down
; with drawn length = 128 + originally intended length.
; To restore the original behaviour, use AND #$7F.
;;;                     and #$7F
                        ldy #40

; We get here everytime when we draw lines that have direction=down,
; e.g. the titan borders. So we always get the chance to
; init zp_inbox_delay with #40 before a cave starts:
                        sty zp_inbox_delay  ; init zp_inbox_delay

_right:                 sty zp_line_step    ; Y = 1 (when direction right) or Y = 40 (when down)
                        tax       ; X = length

                        pla
                        tay       ; Y = adr (lo) of start pos
                        pla
                        pha
                        and #3
                        ora #>(cave_buffer)
                        sta zp_cave_draw+1  ; adr (hi) of start pos

_loop:                  pla
                        pha

        ;;;             and #$7C
        ;;;             lsr
                      ; ALR #$7C   ; illegal opcode, just like AND #$7C followed by LSR
                        .by $4B, $7C

                        lsr
                        ; now A = object and C=0
                        sta (zp_cave_draw),Y

                        tya
                        ; we still have C=0, see above.
                        adc zp_line_step
                        tay

                        bcc _skip_inc
                        inc zp_cave_draw+1
_skip_inc:              dex
                        bne _loop

                        pla   ; bit 7 is set when we have to quit drawing
                        bpl _continue

; restore stack pointer to #$FD so the following RTS fetches the real return adress:
                        ldx #$FD
                        txs
                        rts

; ========================================================================

; Input:
;  A:  = #$A9 when fade to titan
;      = #$B1 when fade in current cave

fade_in::               sta _fetch_object       ; Cool hack. See comment below.

ASSERT <(cave_buffer) = 0
       ;;;              lda #0
       ;;;              sta zp_cave_scan
; We don't need to initialize  zp_cave_scan  with #0 here since it is already #0.
                        lda #>(cave_buffer)
                        sta zp_cave_scan+1

                        ldy #40+40
                        sty zp_rockford_seen_alive   ; Init counter "Rockford seen alive"
_loop:

; Hack: To switch between "fade in current cave" and "fade out to titan"
;       we modify  lda (zp_cave_scan),Y  into  lda #$04  and back.
; $04: fade in/out titan wall
; Since  zp_cave_scan == #$04, we have to modify the command byte only (from $B1 to $A9 and back).
ASSERT zp_cave_scan = $04
_fetch_object:          lda (zp_cave_scan),Y

                        jsr set_object

; draw bottom line of titan border and also fill area after cave buffer:
                        lda #$02   ; $02: titan wall
                        sta adr_titan_border_bottom_left,Y

                        iny
                        bne _loop
ASSERT (cave_buffer+$0400) = $8000
                        inc zp_cave_scan+1      ; End when $8000 reached
                        bpl _loop

                        rts

; ========================================================================
; ========================================================================
; ========================================================================

;  Each command is 3 bytes:
;   b1  =  length (bits 0..5) and direction (bit 7: 0=right; 1=down)
;   b2  =  adr (lo) of start pos
;   b3  =  bits 0..1: adr (hi) of start pos (only bits 0 and 1; you have to add the rest of the msb yourself)
;          bits 2..6: object to draw (2x LSR needed)
;          bit 7: if set, quit drawing after this command

DEFINE DIR_RIGHT 0
DEFINE DIR_DOWN  1

; MACRO DRAW_CMD  object  length  direction  adr  is_last
DEFINE MACRO DRAW_CMD
    .by (@2)+((@3)*128)
    .by <(@4)
    .by ((>(@4))&$03)+((@1)*4)+((@5)*128)
END

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

cave_data_for_stack_src::

; titan border:
                        ; object $02: titan wall
                        MACRO DRAW_CMD  $02  120 DIR_RIGHT (adr_titan_border_top_left-80)  0
                        MACRO DRAW_CMD  $02  21  DIR_DOWN  (adr_titan_border_top_left+40)  0
                        MACRO DRAW_CMD  $02  21  DIR_DOWN  (adr_titan_border_top_left+39)  1

ASSERT MEMORY(cave_data_for_stack_src)<<1 = value_D018

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

cave_data_cave_A:
                        .by 10  ; initial random seed

                        .by 7, 9, 8, 0   ; cave colors

                        .by      9    ; Diamond: threshold 9
                        .by      50   ; Boulder: threshold 50
                        .by      60   ; Space  : threshold 60
                        .by      0    ; -/-

                        .by $10       ; Diamond: object $10: diamond
                        .by $12       ; Boulder: object $12: boulder
                        .by $00       ; Space  : object $00: space
                        .by $00       ; -/-

                        ; object $0F: wall
                        MACRO DRAW_CMD  $0F  30  DIR_RIGHT  (adr_titan_border_top_left+281)  0
                        MACRO DRAW_CMD  $0F  30  DIR_RIGHT  (adr_titan_border_top_left+569)  0

; With random seed #$C4 (i.e. in Level 196), the inbox is totally enclosed by boulders
; so Rockford is trapped right from the start. To avoid this, we place some
; dirt next to the inbox:
                        ; object $01: dirt
                        MACRO DRAW_CMD  $01   1  DIR_RIGHT  (adr_titan_border_top_left+82)   0

                        ; object $19: inbox
                        MACRO DRAW_CMD  $19   1  DIR_RIGHT  (adr_titan_border_top_left+83)   0

                        ; object $1B: outbox
                        MACRO DRAW_CMD  $1B   1  DIR_RIGHT  (adr_titan_border_top_left+678)  1

DEFINE SP_FOR_CAVE_A ((cave_data_cave_A-cave_data_for_stack_src)-1)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

cave_data_cave_B:
                        .by 3^1  ; initial random seed

                        .by 3, 10, 4, 0   ; cave colors

                        .by      2    ; Firefly: threshold 2
                        .by      12   ; Diamond: threshold 12  (Original BD1 threshold is 9 here.)
                        .by      50   ; Boulder: threshold 50
                        .by      60   ; Space  : threshold 60

                        .by $18       ; Firefly: object $18: firefly
                        .by $10       ; Diamond: object $10: diamond
                        .by $12       ; Boulder: object $12: boulder
                        .by $00       ; Space  : object $00: space

                        ; object $0F: wall
                        MACRO DRAW_CMD  $0F  38  DIR_RIGHT  (adr_titan_border_top_left+241)  0
                        MACRO DRAW_CMD  $0F  38  DIR_RIGHT  (adr_titan_border_top_left+521)  0
                        MACRO DRAW_CMD  $0F  20  DIR_DOWN   (adr_titan_border_top_left+48)   0
                        MACRO DRAW_CMD  $0F  20  DIR_DOWN   (adr_titan_border_top_left+56)   0
                        MACRO DRAW_CMD  $0F  20  DIR_DOWN   (adr_titan_border_top_left+64)   0
                        MACRO DRAW_CMD  $0F  20  DIR_DOWN   (adr_titan_border_top_left+72)   0

                        ; object $00: space
                        MACRO DRAW_CMD  $00  38  DIR_RIGHT  (adr_titan_border_top_left+121)  0
                        MACRO DRAW_CMD  $00  38  DIR_RIGHT  (adr_titan_border_top_left+361)  0
                        MACRO DRAW_CMD  $00  38  DIR_RIGHT  (adr_titan_border_top_left+641)  0
                        MACRO DRAW_CMD  $00  20  DIR_DOWN   (adr_titan_border_top_left+60)   0

; With some random seeds, a randomly placed Firefly is immediately next to the inbox
; so Rockford is killed instantly when the inbox opens, or
; the outbox is blocked by falling boulders and there is no way for Rockford
; to prevent this from happening.
; To avoid this, we place some dirt around the inbox:
                        ; object $01: dirt
                        MACRO DRAW_CMD  $01   3  DIR_RIGHT  (adr_titan_border_top_left+737)  0
                        MACRO DRAW_CMD  $01   3  DIR_RIGHT  (adr_titan_border_top_left+777)  0

; With random seed #$83 (i.e. in Level 131), a firefly explodes near the inbox
; and thus exposes the inbox to a second firefly that is coming along,
; but Rockford can escape by immediately running to the left and then up.
; (So we don't have to tweak the cave data any further.)

                        ; object $19: inbox
                        MACRO DRAW_CMD  $19   1  DIR_RIGHT  (adr_titan_border_top_left+778)  0

                        ; object $1B: outbox
                        MACRO DRAW_CMD  $1B   1  DIR_RIGHT  (adr_titan_border_top_left+818)  1

DEFINE SP_FOR_CAVE_B ((cave_data_cave_B-cave_data_for_stack_src)-1)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

cave_data_cave_C:
                        .by 0^1  ; initial random seed

                        .by 13, 6, 14, 0   ; cave colors

                        .by      9    ; Diamond: threshold 9
                        .by      50   ; Boulder: threshold 50
                        .by      100  ; Wall   : threshold 100
                        .by      0    ; -/-

                        .by $10       ; Diamond: object $10: diamond
                        .by $12       ; Boulder: object $12: boulder
                        .by $0F       ; Wall   : object $0F: wall
                        .by $00       ; -/-

; With several random seeds, the inbox is totally enclosed or blocked by
; boulders so Rockford is trapped right from the start. To avoid this, we
; place some dirt next to the inbox:
                        ; object $01: dirt
                        MACRO DRAW_CMD  $01   4  DIR_RIGHT  (adr_titan_border_top_left+82)   0

                        ; object $19: inbox
                        MACRO DRAW_CMD  $19   1  DIR_RIGHT  (adr_titan_border_top_left+83)   0

; With several random seeds, the outbox is totally enclosed or blocked by
; boulders so Rockford can never reach it. To avoid this, we place some
; dirt next to the outbox:
                        ; object $01: dirt
                        MACRO DRAW_CMD  $01   3  DIR_RIGHT  (adr_titan_border_top_left+755)  0

                        ; object $1B: outbox
                        MACRO DRAW_CMD  $1B   1  DIR_RIGHT  (adr_titan_border_top_left+758)  1

DEFINE SP_FOR_CAVE_C ((cave_data_cave_C-cave_data_for_stack_src)-1)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

cave_data_cave_D:
                        .by 0^1  ; initial random seed

                        .by 15, 2, 10, 0   ; cave colors

                        .by      30   ; Space  : threshold 30
                        .by      33   ; Firefly: threshold 33
                        .by      41   ; Diamond: threshold 41
                        .by      80   ; Boulder: threshold 80

                        .by $00       ; Space  : object $00: space
                        .by $18       ; Firefly: object $18: firefly
                        .by $10       ; Diamond: object $10: diamond
                        .by $12       ; Boulder: object $12: boulder

; With some random seeds, a randomly placed Firefly is immediately next to the inbox
; so Rockford is killed instantly when the inbox opens. To avoid this, we place some
; dirt around the inbox:
                        ; object $01: dirt
                        MACRO DRAW_CMD  $01   3  DIR_DOWN   (adr_titan_border_top_left+683)   0
                        MACRO DRAW_CMD  $01   3  DIR_RIGHT  (adr_titan_border_top_left+722)   0

                        ; object $19: inbox
                        MACRO DRAW_CMD  $19   1  DIR_RIGHT  (adr_titan_border_top_left+723)   0

                        ; object $1B: outbox
                        MACRO DRAW_CMD  $1B   1  DIR_RIGHT  (adr_titan_border_top_left+118)   1

DEFINE SP_FOR_CAVE_D ((cave_data_cave_D-cave_data_for_stack_src)-1)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

cave_data_for_stack_src_end::

DEFINE cave_data_for_stack_length (cave_data_for_stack_src_end-cave_data_for_stack_src)
ASSERT cave_data_for_stack_length < INIT_MAX_COPY

; ========================================================================
; ========================================================================
; ========================================================================

; MACRO for easy copying of several bytes to a destination address:

; COPY_BYTES source_adr number_of_bytes dest_adr
DEFINE MACRO COPY_BYTES
  COPY (@1) ((@1)+(@2)) (@3)
END

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

zp_data:
PC (PC+zp_copy_length)

        MACRO COPY_BYTES zp_copy_to  zp_copy_length  zp_data

; ========================================================================
; ========================================================================
; ========================================================================

file_end:

DEFINE file_size (file_end-(file_start-2))

DEFINE BYTES_LEFT_UP_TO_1026   1026-file_size
PRINT BYTES_LEFT_UP_TO_1026

; Assert that file size is 1026 bytes or less:
ASSERT file_size <= 1026

; ========================================================================

PRINT POKE_8_INTO_HERE_FOR_CHEAT

; ========================================================================

PRINT game_start

; ========================================================================

; Now that everything is assembled, save to file:
SAVE file_start-2 file_end 1k-mini-bdash.prg
